/**
 *
 */
package gov.va.med.mhv.usermgmt.integration.adapter;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.tigris.atlas.messages.MessageUtils;
import org.tigris.atlas.service.VoidServiceResponse;

import gov.va.med.mhv.core.util.Precondition;
import gov.va.med.mhv.integration.registry.transfer.Status;
import gov.va.med.mhv.integration.registry.transfer.StatusType;
import gov.va.med.mhv.usermgmt.integration.service.delegate.IntegrationServiceDelegateFactory;
import gov.va.med.mhv.usermgmt.integration.service.delegate.PatientCorrelationServiceDelegate;
import gov.va.med.mhv.usermgmt.integration.service.util.RegistryInterfaceUtils;
import gov.va.med.mhv.usermgmt.transfer.Patient;
import gov.va.med.mhv.usermgmt.util.MessageKeys;

/**
 * @author Rob Proper (Aquilent Inc.)
 *
 */
public final class PatientCorrelationServiceAdapter {

    private static final Log LOG = LogFactory.getLog(
        PatientCorrelationServiceAdapter.class);

    public VoidServiceResponse correlate(Patient patient) {
        Precondition.assertNotNull("patient", patient);
        Status status = createDelegate().correlatePatient(
            RegistryInterfaceUtils.toPatient(patient));
        return createResponse("correlate", status, patient);
    }

    public VoidServiceResponse uncorrelate(Patient patient) {
        Precondition.assertNotNull("patient", patient);
        Status status = createDelegate().uncorrelatePatient(
            RegistryInterfaceUtils.toPatient(patient));
        return createResponse("uncorrelate", status, patient);
    }

    private VoidServiceResponse createResponse(String action, Status status,
        Patient patient)
    {
        VoidServiceResponse response = new VoidServiceResponse();
        if (((status != null) && StatusType.Error.equals(status.getStatus()))) {
            LOG.error(action + " " + describe(patient) + " failed, because " +
                status.getStatusDescription() + ".");
            response.getMessages().addMessage(MessageUtils.createErrorMessage(
                getErrorKey(status), null, null));
        } else if (LOG.isDebugEnabled()) {
            LOG.debug(action + " " + describe(patient) + " succeeded");
        }
        return response;
    }

    private String getErrorKey(Status status) {
        return ((status != null) &&
        		(StringUtils.contains(status.getStatusDescription(),
        				"Production is not running")
        		 || StringUtils.contains(status.getStatusDescription(),
        		 		"java.net.ConnectException")))
            ? MessageKeys.ENSEMBLE_ACCESS_PROBLEM_AT_IPA
            : MessageKeys.UNKNOWN_EXCEPTION_OCCURRED;
    }

    private PatientCorrelationServiceDelegate createDelegate() {
        return IntegrationServiceDelegateFactory.
            createPatientCorrelationServiceDelegate();
    }

    private String describe(Patient patient) {
        if (patient == null) {
            return "<unknown patient>";
        }
        StringBuilder builder = new StringBuilder();
        builder.append("Patient[");
        builder.append("icn=").append(patient.getIcn());
        builder.append(";user_name=");
        builder.append((patient.getUserProfile() != null)
            ? patient.getUserProfile().getUserName()
            : "<no user profile>");
        builder.append("]");
        return builder.toString();
    }
}
